////////////////////////////////////////////////////////////////////////////////
// Naam:       4in105.ino                                                     //
//             http://robotigs.nl/robots/includes/parts.php?idpart=364        //
//                                                                            //
// Created by: HARB rboek2@gmail.com may 2021 GPL copyrights                  //
// Origineel:  https://github.com/sidddy/flora                                //
//----------------------------------------------------------------------------//
// Board:            ESP32 Dev Module                                         //
// Partition Scheme: No OTA (2MB APP/2MB SPIFFS)                              //
//----------------------------------------------------------------------------//
// As outputs the following modules are mounted:                              //
// - Standard Onboard LED                                                     //
//           http://robotigs.nl/robots/includes/parts.php?idpart=185          //
// - OLED 128X64 monochrome                                                   //
//           http://robotigs.nl/robots/includes/parts.php?idpart=185          //
//                                                                            //
// As inputs the following modules are mounted:                               //
//                                                                            //
// For communications and statistics are mounted:                             //
// - Standard Serial Monitor output                                           //
//            http://robotigs.nl/robots/includes/parts.php?idpart=43          //
////////////////////////////////////////////////////////////////////////////////


// SET PRECOMPILER OPTIONS *****************************************************
  //Define VARIABLE SETTINGS ---------------------------------------------------
  char* FLORA_DEVICES[] = {                        //Array of 4in1 MAC addresses
      "C4:7C:8D:6B:5F:5C"                                              //4in1wit
      , "80:EA:CA:89:4D:16"                                          //4in1groen
      };
  #define SLEEP_DURATION 30 * 60               //Sleep between 2 runs in seconds
  #define EMERGENCY_HIBERNATE 3 * 60  //Emergency hibernate countdown in seconds
  #define BATTERY_INTERVAL 6           //How often read the battery in run count
  #define RETRY 3       //How often retry a device in a run when something fails
  const char*   WIFI_SSID       = "H369AAA781A";
  const char*   WIFI_PASSWORD   = "DD7D5CAAA3DE";
  const char*   MQTT_HOST       = "192.168.2.24";
  const int     MQTT_PORT       = 1883;
  const char*   MQTT_CLIENTID   = "miflora-client";
  const char*   MQTT_USERNAME   = "username";
  const char*   MQTT_PASSWORD   = "password";
  const String  MQTT_BASE_TOPIC = "/4in1"; 
  const int     MQTT_RETRY_WAIT = 5000;

  //Define the needed header files for the precompiler, no charge if not used --
  #include "BLEDevice.h"
  #include <WiFi.h>
  #include <PubSubClient.h>
  #include <Wire.h>                             //Standaard geinstalleerd in IDE
  #include <Adafruit_GFX.h>      //Standaard meegeleverd met IDE, zelf activeren
  #include <Adafruit_SSD1306.h>  //Standaard meegeleverd met IDE, zelf activeren

  //Define PINS ----------------------------------------------------------------
  #define OLED_RESET    -1                   //Reset pin# or -1 if not available

  //Define EEPROM variables ----------------------------------------------------
  //Define DATABASE VARIABLES --------------------------------------------------

  //Define variables -----------------------------------------------------------
  #define SCREEN_WIDTH 128                       //OLED display width, in pixels
  #define SCREEN_HEIGHT 64                      //OLED display height, in pixels


  


RTC_DATA_ATTR int bootCount = 0; //Boot count used to check if battery status should be read
static int deviceCount = sizeof FLORA_DEVICES / sizeof FLORA_DEVICES[0]; //Device count
static BLEUUID serviceUUID("00001204-0000-1000-8000-00805f9b34fb"); //The remote service we wish to connect to
static BLEUUID uuid_version_battery("00001a02-0000-1000-8000-00805f9b34fb"); //The characteristic of the remote service we are interested in
static BLEUUID uuid_sensor_data("00001a01-0000-1000-8000-00805f9b34fb");
static BLEUUID uuid_write_mode("00001a00-0000-1000-8000-00805f9b34fb");


  //Initialize OBJECTS ---------------------------------------------------------
  Adafruit_SSD1306 display(SCREEN_WIDTH, SCREEN_HEIGHT, &Wire, OLED_RESET);
  TaskHandle_t hibernateTaskHandle = NULL;

WiFiClient espClient;
PubSubClient client(espClient);

void connectWifi() {
  Serial.println("Connecting to WiFi...");
  WiFi.begin(WIFI_SSID, WIFI_PASSWORD);

  while (WiFi.status() != WL_CONNECTED) {
    delay(500);
    Serial.print(".");
  }

  Serial.println("");
  Serial.println("WiFi connected");
  Serial.println("");
}

void disconnectWifi() {
  WiFi.disconnect(true);
  Serial.println("WiFi disonnected");
}

void connectMqtt() {
  Serial.println("Connecting to MQTT...");
  client.setServer(MQTT_HOST, MQTT_PORT);

  while (!client.connected()) {
    if (!client.connect(MQTT_CLIENTID, MQTT_USERNAME, MQTT_PASSWORD)) {
      Serial.print("MQTT connection failed:");
      Serial.print(client.state());
      Serial.println("Retrying...");
      delay(MQTT_RETRY_WAIT);
    }
  }

  Serial.println("MQTT connected");
  Serial.println("");
}

void disconnectMqtt() {
  client.disconnect();
  Serial.println("MQTT disconnected");
}

BLEClient* getFloraClient(BLEAddress floraAddress) {
  BLEClient* floraClient = BLEDevice::createClient();

  if (!floraClient->connect(floraAddress)) {
    Serial.println("- Connection failed, skipping");
    return nullptr;
  }

  Serial.println("- Connection successful");
  return floraClient;
}

BLERemoteService* getFloraService(BLEClient* floraClient) {
  BLERemoteService* floraService = nullptr;

  try {
    floraService = floraClient->getService(serviceUUID);
  }
  catch (...) {
    // something went wrong
  }
  if (floraService == nullptr) {
    Serial.println("- Failed to find data service");
  }
  else {
    Serial.println("- Found data service");
  }

  return floraService;
}

bool forceFloraServiceDataMode(BLERemoteService* floraService) {
  BLERemoteCharacteristic* floraCharacteristic;

  Serial.println("- Force device in data mode");  //Get device mode characteristic, needs to be changed to read data
  floraCharacteristic = nullptr;
  try {
    floraCharacteristic = floraService->getCharacteristic(uuid_write_mode);
  }
  catch (...) {     //Something went wrong

  }
  if (floraCharacteristic == nullptr) {
    Serial.println("-- Failed, skipping device");
    return false;
  }

  // write the magic data
  uint8_t buf[2] = {0xA0, 0x1F};
  floraCharacteristic->writeValue(buf, 2, true);

  delay(500);
  return true;
}

bool readFloraDataCharacteristic(BLERemoteService* floraService, String baseTopic) {
  BLERemoteCharacteristic* floraCharacteristic = nullptr;

  // get the main device data characteristic
  Serial.println("- Access characteristic from device");
  try {
    floraCharacteristic = floraService->getCharacteristic(uuid_sensor_data);
  }
  catch (...) {
    // something went wrong
  }
  if (floraCharacteristic == nullptr) {
    Serial.println("-- Failed, skipping device");
    return false;
  }

  // read characteristic value
  Serial.println("- Read value from characteristic");
  std::string value;
  try{
    value = floraCharacteristic->readValue();
  }
  catch (...) {
    // something went wrong
    Serial.println("-- Failed, skipping device");
    return false;
  }
  const char *val = value.c_str();

  Serial.print("Hex: ");
  for (int i = 0; i < 16; i++) {
    Serial.print((int)val[i], HEX);
    Serial.print(" ");
  }
  Serial.println(" ");

  int16_t* temp_raw = (int16_t*)val;
  float temperature = (*temp_raw) / ((float)10.0);
  Serial.print("-- Temperatuur: ");
  Serial.println(temperature);

  int moisture = val[7];
  Serial.print("-- Vochtigheid: ");
  Serial.println(moisture);

  int light = val[3] + val[4] * 256;
  Serial.print("-- Lichtkracht: ");
  Serial.println(light);
 
  int conductivity = val[8] + val[9] * 256;
  Serial.print("-- Geleiding: ");
  Serial.println(conductivity);

  if (temperature > 200 or temperature < -200) {
    Serial.println("-- Unreasonable values received, skip publish");
    return false;
  }

  char buffer[64];

  snprintf(buffer, 64, "%f", temperature);
  client.publish((baseTopic + "temperatuur").c_str(), buffer); 
  snprintf(buffer, 64, "%d", moisture); 
  client.publish((baseTopic + "vochtigheid").c_str(), buffer);
  snprintf(buffer, 64, "%d", light);
  client.publish((baseTopic + "lichtkracht").c_str(), buffer);
  snprintf(buffer, 64, "%d", conductivity);
  client.publish((baseTopic + "geleiding").c_str(), buffer);

  //Test hardware and software -------------------------------------------------
  display.setCursor(0,0);                             //Start at top-left corner
  display.println(temperature);
  display.println(moisture);
   display.println(light);
  display.println(conductivity);
  display.display();                     //Show the display buffer on the screen

  return true;
}

bool readFloraBatteryCharacteristic(BLERemoteService* floraService, String baseTopic) {
  BLERemoteCharacteristic* floraCharacteristic = nullptr;

  // get the device battery characteristic
  Serial.println("- Access battery characteristic from device");
  try {
    floraCharacteristic = floraService->getCharacteristic(uuid_version_battery);
  }
  catch (...) {
    // something went wrong
  }
  if (floraCharacteristic == nullptr) {
    Serial.println("-- Failed, skipping battery level");
    return false;
  }

  // read characteristic value
  Serial.println("- Read value from characteristic");
  std::string value;
  try{
    value = floraCharacteristic->readValue();
  }
  catch (...) {
    // something went wrong
    Serial.println("-- Failed, skipping battery level");
    return false;
  }
  const char *val2 = value.c_str();
  int battery = val2[0];

  char buffer[64];

  Serial.print("-- Batterij: ");
  Serial.println(battery);
  snprintf(buffer, 64, "%d", battery);
  client.publish((baseTopic + "batterij").c_str(), buffer);

  return true;
}

bool processFloraService(BLERemoteService* floraService, char* deviceMacAddress, bool readBattery) {
  // set device in data mode
  if (!forceFloraServiceDataMode(floraService)) {
    return false;
  }

  String baseTopic = MQTT_BASE_TOPIC + "/" + deviceMacAddress + "/";
  bool dataSuccess = readFloraDataCharacteristic(floraService, baseTopic);

  bool batterySuccess = true;
  if (readBattery) {
    batterySuccess = readFloraBatteryCharacteristic(floraService, baseTopic);
  }

  return dataSuccess && batterySuccess;
}

bool processFloraDevice(BLEAddress floraAddress, char* deviceMacAddress, bool getBattery, int tryCount) {
  Serial.print("Processing Flora device at ");
  Serial.print(floraAddress.toString().c_str());
  Serial.print(" (try ");
  Serial.print(tryCount);
  Serial.println(")");

  // connect to flora ble server
  BLEClient* floraClient = getFloraClient(floraAddress);
  if (floraClient == nullptr) {
    return false;
  }

  // connect data service
  BLERemoteService* floraService = getFloraService(floraClient);
  if (floraService == nullptr) {
    floraClient->disconnect();
    return false;
  }

  // process devices data
  bool success = processFloraService(floraService, deviceMacAddress, getBattery);

  // disconnect from device
  floraClient->disconnect();

  return success;
}

void hibernate() {
  esp_sleep_enable_timer_wakeup(SLEEP_DURATION * 1000000ll);
  Serial.println("Going to sleep now.");
  delay(100);
  esp_deep_sleep_start();
}

void delayedHibernate(void *parameter) {
  delay(EMERGENCY_HIBERNATE*1000); // delay for five minutes
  Serial.println("Something got stuck, entering emergency hibernate...");
  hibernate();
}

void setup() {
  // all action is done when device is woken up
  Serial.begin(115200);
  delay(1000);

  
  //Start display --------------------------------------------------------------
  if(!display.begin(SSD1306_SWITCHCAPVCC, 0x3C)) { //Generate Vdc intr from 3.3V
    Serial.println(F("SSD1306 allocation failed"));    //Oeps, hij doet het niet
    for(;;);                                       //Don't proceed, loop forever
  } //End of if(!display.begin(SSD1306_SWITCHCAPVCC, 0x3C))
  display.clearDisplay();                                     //Clear the buffer
  display.setTextSize(2);                           //1=onleesbaar klein 2=groot
  display.setTextColor(WHITE);                                 //Draw white text

  // increase boot count
  bootCount++;

  // create a hibernate task in case something gets stuck
  xTaskCreate(delayedHibernate, "hibernate", 4096, NULL, 1, &hibernateTaskHandle);

  Serial.println("Initialize BLE client...");
  BLEDevice::init("");
  BLEDevice::setPower(ESP_PWR_LVL_P7);

  // connecting wifi and mqtt server
  connectWifi();
  connectMqtt();

  // check if battery status should be read - based on boot count
  bool readBattery = ((bootCount % BATTERY_INTERVAL) == 0);

  // process devices
  for (int i=0; i<deviceCount; i++) {
    int tryCount = 0;
    char* deviceMacAddress = FLORA_DEVICES[i];
    BLEAddress floraAddress(deviceMacAddress);

    while (tryCount < RETRY) {
      tryCount++;
      if (processFloraDevice(floraAddress, deviceMacAddress, readBattery, tryCount)) {
        break;
      }
      delay(1000);
    }
    delay(1500);
  }

  //Test hardware and software -------------------------------------------------
  display.setCursor(0,0);                             //Start at top-left corner
  display.println("Ok");
  display.display();                     //Show the display buffer on the screen

  // disconnect wifi and mqtt
  disconnectWifi();
  disconnectMqtt();

  // delete emergency hibernate task
  vTaskDelete(hibernateTaskHandle);

  // go to sleep now
  hibernate();
}

void loop() {
  /// we're not doing anything in the loop, only on device wakeup
  delay(10000);
}
